<?php

namespace Ptb\Sso\Saml\Http\Controllers;

use Illuminate\Http\Request;
use OneLogin\Saml2\Error;
use OneLogin\Saml2\ValidationError;
use Ptb\Sso\Saml\Events\SamlLoginEvent;
use Ptb\Sso\Saml\Events\SamlLogoutEvent;
use Ptb\Sso\Saml\Saml;

class SamlController
{
    /**
     * @param Saml $saml
     * @throws Error
     */
    public function index(Saml $saml)
    {
        $saml->login(
            returnTo: route(config('saml.loginRoute'))
        );
    }

    /**
     * @throws Error
     */
    public function show(Saml $saml)
    {
        return response(
            content: $saml->getMetadata(),
            status: 200,
            headers: [
                'Content-Type' => 'text/xml',
            ]
        );
    }

    /**
     * @throws Error
     * @throws ValidationError
     */
    public function store(Saml $saml)
    {
        if ($errors = $saml->acs()) {
            if (count($errors) > 0) {
                logger()->error(
                    message: 'SAML ACS Error Reason ',
                    context: [
                        'error' => $saml->getLastErrorReason()
                    ]
                );

                logger()->error(
                    message: 'SAML ACS Error(s)',
                    context: $errors
                );

                return redirect(config('saml.errorRoute'));
            }
        }

        event(new SamlLoginEvent($saml));

        if ($redirectUrl = $saml->getIntendedUrl()) {
            return redirect($redirectUrl);
        } else {
            return redirect('/');
        }
    }

    /**
     * @return void
     */
    public function destroy(Saml $saml, Request $request)
    {
        event(new SamlLogoutEvent());

        try {
            $returnTo = $request->get('returnTo', route(config('saml.logoutRoute')));
            $sessionIndex = $request->get('sessionIndex');
            $nameId = $request->get('nameId');
            $saml->logout($returnTo, [], $nameId, $sessionIndex);
        } catch (Error $error) {
            // do nothing
        }
    }

    /**
     * @throws Error
     */
    public function forceDelete(Saml $saml)
    {
        $errors = $saml->sls(config( 'saml.retrieveParametersFromServer'));
        if (!empty($errors)) {
            logger()->error('SAML SLS Error(s)', $errors);
        }

        return redirect(route(config('saml.logoutRoute')));
    }
}
