<?php

namespace Ptb\Pace\Services;

use Closure;
use SoapFault;
use Throwable;
use Ptb\Pace\Service;
use Ptb\Pace\Soap\SoapClient;
use Ptb\Pace\Soap\Middleware\Transaction;
use Ptb\Pace\Soap\Middleware\StartTransaction;
use Ptb\Pace\Soap\Middleware\CommitTransaction;
use Ptb\Pace\Soap\Middleware\RollbackTransaction;

class TransactionService extends Service
{
    /**
     * Wrap the specified closure in a transaction.
     *
     * @param Closure $callback
     */
    public function transaction(Closure $callback)
    {
        $this->startTransaction();

        try {
            $callback();
        } catch (SoapFault $exception) {
            // Pace has already rolled the transaction back.
            SoapClient::removeMiddleware('transaction');
            throw $exception;
        } catch (Throwable $exception) {
            $this->rollback();
            throw $exception;
        }

        $this->commit();
    }

    /**
     * Start a new transaction.
     *
     * @param int $timeout
     */
    public function startTransaction(int $timeout = 60)
    {
        SoapClient::addMiddleware('startTransaction', new StartTransaction);

        $response = $this->soap->startTransaction(['in0' => $timeout]);

        SoapClient::removeMiddleware('startTransaction');
        SoapClient::addMiddleware('transaction', new Transaction($response->out));
    }

    /**
     * Rollback the transaction.
     */
    public function rollback()
    {
        SoapClient::addMiddleware('rollback', new RollbackTransaction);

        $this->soap->rollback();

        SoapClient::removeMiddleware('rollback');
        SoapClient::removeMiddleware('transaction');
    }

    /**
     * Commit the transaction.
     */
    public function commit()
    {
        SoapClient::addMiddleware('commit', new CommitTransaction);

        $this->soap->commit();

        SoapClient::removeMiddleware('commit');
        SoapClient::removeMiddleware('transaction');
    }
}
