<?php

namespace Ptb\Pace\Actions;

use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Ptb\Pace\Actions\ChangeOrders\AddPostageAdvance;
use Ptb\Pace\Actions\Jobs\AddMailing;
use Ptb\Pace\Data\Attachment;
use Ptb\Pace\Data\Estimate;
use Ptb\Pace\Data\JobPart;
use Ptb\Pace\Data\JobPartMail;
use Ptb\Pace\Data\JobType;
use Ptb\Pace\Data\SubJobType;
use Ptb\Pace\Enums\TaskStatus;
use Ptb\Pace\Models\PaceTask;
use Ptb\Pace\Models\PaceTaskPart;
use Ptb\Pace\Requests\CalculateEstimateRequest;
use Ptb\Pace\Requests\ConvertEstimateToJobRequest;
use Ptb\Pace\Requests\GetEstimateConvertToJobRequest;
use Throwable;

final class ConvertTaskToJob
{
    public function __construct(
        private readonly CalculateEstimateRequest       $calculateEstimateRequest,
        private readonly GetEstimateConvertToJobRequest $getEstimateConvertToJobRequest,
        private readonly ConvertEstimateToJobRequest    $convertEstimateToJobRequest,
        private readonly AddPostageAdvance              $addPostageAdvance,
        private readonly AddMailing                     $addMailing,
    )
    {
    }

    public function handle(
        PaceTask $task,
        array    $option = [],
    )
    {
        try {
            $estimate = Estimate::filterByNumber(estimateNumber: $task->task_estimate_number);

            // before we can convert to a job the estimate
            // has to be calculated
            $this->calculateEstimateRequest->handle(estimate: $estimate);
            $convertToJob = $this->getEstimateConvertToJobRequest->handle(estimate: $estimate);

            $lead_time = $task->parts->max('lead_time');

            $convertToJob->jobType = JobType::from($task->task_job_type_id);
            $convertToJob->subJobType = SubJobType::from($task->task_sub_job_type_id);
            $convertToJob->jobOrderType = $task->task_job_order_type;
            $convertToJob->description = $task->task_description;
            $convertToJob->promiseDate = Carbon::now()->addDays($lead_time);

            $job = $this->convertEstimateToJobRequest->handle(estimateToJob: $convertToJob);

            $task->update([
                'task_job_number' => $job->job,
                'task_status' => TaskStatus::JobCreated,
            ]);

            /**
             * @var JobPart[] $job_parts
             * @var JobPart $job_part
             */
            $job_parts = JobPart::filterByJob(job: $job->job);
            foreach ($job_parts as $job_part) {
                $task_part = $task->parts()
                    ->whereHas('automation', function ($q) use ($job_part) {
                        return $q->where('product_type_id',$job_part->jobProductType);
                    })
                    ->first();

                if (!empty($task_part)) {
                    if (!empty($part->part_mail_reports_location)) {
                        $this->handleMailingSetup(
                            part: $task_part,
                            jobPart: $job_part
                        );

                        $this->addPostageAdvance->handle(
                            jobPart: $job_part,
                            postageAmount: $task_part->part_postage_amount,
                        );
                    }
                }
            }
        } catch (Throwable $fault) {
            throw $fault;
        }
    }

    /**
     * @param PaceTaskPart $part
     * @param JobPart $jobPart
     * @return void
     */
    protected function handleMailingSetup(PaceTaskPart $part, JobPart $jobPart): void
    {
        $jobPartMail = JobPartMail::forJobPart(jobPart: $jobPart);
        $jobPartMail->addressFormat = $part->automation->mailing->address_format;
        $jobPartMail->aspectRatio = $part->automation->mailing->aspect_ratio;
        $jobPartMail->barcode = $part->automation->mailing->barcode;
        $jobPartMail->cassList = $part->automation->mailing->cass_list;
        $jobPartMail->cleanList = $part->automation->mailing->clean_list;
        $jobPartMail->coMingle = $part->automation->mailing->co_mingle;
        $jobPartMail->duplicateAddress = $part->automation->mailing->duplicate_address;
        $jobPartMail->extraStock = $part->automation->mailing->extra_stock;
        $jobPartMail->deDupeType = $part->automation->mailing->de_dupe_type;
        $jobPartMail->insertType = $part->automation->mailing->insert_type;
        $jobPartMail->instructions = $part->automation->mailing->instructions;
        $jobPartMail->listSource = $part->automation->mailing->list_source;
        $jobPartMail->mailingClass = $part->automation->mailing->mailing_class;
        $jobPartMail->mailingPreparedFor = $part->automation->mailing->mailing_prepared_for;
        $jobPartMail->mailingSize = $part->automation->mailing->mailing_size;
        $jobPartMail->mailingTabs = $part->automation->mailing->mailing_tabs;
        $jobPartMail->mailingType = $part->automation->mailing->mailing_type;
        $jobPartMail->matchedMailing = $part->automation->mailing->matched_mailing;
        $jobPartMail->ncoaList = $part->automation->mailing->ncoa_list;
        $jobPartMail->permitBilling = $part->automation->mailing->permit_billing;
        $jobPartMail->permitPrint = $part->automation->mailing->permit_print;
        $jobPartMail->permitUsed = $part->automation->mailing->permit_used;
        $jobPartMail->postageType = $part->automation->mailing->postage_type;
        $jobPartMail->printCase = $part->automation->mailing->print_case;
        $jobPartMail->printType = $part->automation->mailing->print_type;
        $jobPartMail->processed = true;
        $jobPartMail->removeDuplicateAddress = $part->automation->mailing->remove_duplicate_address;
        $jobPartMail->removeInternational = $part->automation->mailing->remove_international;
        $jobPartMail->removeUndeliverable = $part->automation->mailing->remove_undeliverable;
        $jobPartMail->returnAddress = $part->automation->mailing->return_address;
        $jobPartMail->returnPrint = $part->automation->mailing->return_print;

        $this->addMailing->handle(
            part: $jobPart,
            mail: $jobPartMail,
            attachment: Attachment::from([
                'attachment' => Storage::disk('public')->get($part->part_mail_reports_location),
                'name' => basename($part->part_mail_reports_location),
                'description' => $part->part_description
            ])
        );
    }
}