<?php

namespace Ptb\Pace\Concerns;

use Illuminate\Support\Facades\Hash;
use Pace\XPath\Builder;
use Ptb\Pace\PaceConnector;
use Spatie\LaravelData\DataCollection;
use Throwable;

trait HasApiMethods
{
    protected static array $irregularNames = [
        'apSetup' => 'APSetup',
        'arSetup' => 'ARSetup',
        'crmSetup' => 'CRMSetup',
        'crmStatus' => 'CRMStatus',
        'crmUser' => 'CRMUser',
        'csr' => 'CSR',
        'dsfMediaSize' => 'DSFMediaSize',
        'dsfOrderStatus' => 'DSFOrderStatus',
        'faSetup' => 'FASetup',
        'glAccount' => 'GLAccount',
        'glAccountBalance' => 'GLAccountBalance',
        'glAccountBalanceSummary' => 'GLAccountBalanceSummary',
        'glAccountBudget' => 'GLAccountBudget',
        'glAccountingPeriod' => 'GLAccountingPeriod',
        'glBatch' => 'GLBatch',
        'glDepartment' => 'GLDepartment',
        'glDepartmentLocation' => 'GLDepartmentLocation',
        'glJournalEntry' => 'GLJournalEntry',
        'glJournalEntryAudit' => 'GLJournalEntryAudit',
        'glLocation' => 'GLLocation',
        'glRegisterNumber' => 'GLRegisterNumber',
        'glSchedule' => 'GLSchedule',
        'glScheduleLine' => 'GLScheduleLine',
        'glSetup' => 'GLSetup',
        'glSplit' => 'GLSplit',
        'glSummaryName' => 'GLSummaryName',
        'jmfReceivedMessage' => 'JMFReceivedMessage',
        'jmfReceivedMessagePartition' => 'JMFReceivedMessagePartition',
        'jmfReceivedMessageTransaction' => 'JMFReceivedMessageTransaction',
        'jmfReceivedMessageTransactionPartition' => 'JMFReceivedMessageTransactionPartition',
        'poSetup' => 'POSetup',
        'poStatus' => 'POStatus',
        'rssChannel' => 'RSSChannel',
        'uom' => 'UOM',
        'uomDimension' => 'UOMDimension',
        'uomRange' => 'UOMRange',
        'uomSetup' => 'UOMSetup',
        'uomType' => 'UOMType',
        'wipCategory' => 'WIPCategory',
    ];

    /**
     * Create instance of data object from api result with integer id
     *
     * @param int|null $key
     * @return static|null
     */
    public static function fromInteger(int $key = null): static|null
    {
        if (empty($key)) {
            return null;
        }

        try {
            $cache_key = static::$apiObjectName . '-' . md5($key);
            $data = cache()->remember($cache_key, now()->addSeconds(static::getCacheExpire()),function () use ($key) {
                return @app(PaceConnector::class)->readObject(
                    object: static::getObjectName(),
                    key: $key,
                );
            });

            return self::from($data);
        } catch (Throwable $e) {
            report($e);

            return self::from([]);
        }
    }

    /**
     * @return string
     */
    protected static function getObjectName(): string
    {
        return static::$irregularNames[static::$apiObjectName] ?? ucfirst(static::$apiObjectName);
    }

    /**
     * Create instance of data object from api result with string id
     *
     * @param string|null $key
     * @return static|null
     */
    public static function fromString(string $key = null): static|null
    {

        if (empty($key)) {
            return null;
        }

        try {
            $cache_key = static::$apiObjectName . '-' . md5($key);
            $data = cache()->remember($cache_key, now()->addSeconds(static::getCacheExpire()),function () use ($key) {
                return @app(PaceConnector::class)->readObject(
                    object: static::getObjectName(),
                    key: $key,
                );
            });

            return self::from($data);
        } catch (Throwable $e) {
            report($e);

            return self::from([]);
        }
    }

    /**
     * Create a new object
     *
     * @param mixed $data
     * @return mixed
     */
    public static function create(self $data): static
    {
        $pace = @app(PaceConnector::class);
        $result = $pace->createObject(
            object: static::getObjectName(),
            attributes: $data->toArray()
        );

        return static::from($result ?? []);
    }

    /**
     * Update object
     *
     * @param mixed $data
     * @return static
     */
    public static function update(self $data): static
    {
        $result = null;

        $pace = @app(PaceConnector::class);
        $pace->transaction(function () use ($pace, $data, &$result) {
            $result = $pace->updateObject(
                object: static::getObjectName(),
                attributes: $data->toArray()
            );
        });

        return static::from($result ?? []);
    }

    /**
     * @param HasApiMethods $original
     * @param HasApiMethods $data
     * @param mixed|null $newKey
     * @return mixed
     */
    public static function copy(self $original, self $data, mixed $newKey = null)
    {
        $result = null;

        $pace = @app(PaceConnector::class);
        $pace->transaction(function () use ($pace, $original, $data, $newKey, &$result) {
            $result = $pace->cloneObject(
                object: static::getObjectName(),
                attributes: $original->toArray(),
                newAttributes: $data->toArray(),
                newKey: $newKey,
            );
        });

        return static::from($result ?? []);
    }

    /**
     * Upload an attachment
     *
     * @param string $key
     * @param string $field
     * @param string $name
     * @param mixed $content
     * @return mixed
     */
    public static function attach(string $key, string $field, string $name, mixed $content): mixed
    {
        $result = null;

        $pace = @app(PaceConnector::class);
        $pace->transaction(function () use ($pace, $key, $field, $name, $content, &$result) {
            $result = $pace->attachment()->add(
                object: static::getObjectName(),
                key: $key,
                field: $field,
                name: $name,
                content: $content
            );
        });

        return $result;
    }

    /**
     * @return Builder
     */
    protected static function getDefaultFilter(): Builder
    {
        $builder = static::getXPathBuilder();
        $keys = static::getPrimaryKey();

        foreach ($keys as $key) {
            $builder->filter($key, '!=', 0);
        }

        return $builder;
    }

    /**
     * Retrieve a new instance of the XPath Builder
     * @return Builder
     */
    protected static function getXPathBuilder(): Builder
    {
        return new Builder();
    }

    /**
     * @return string[]
     */
    protected static function getPrimaryKey(): array
    {
        $key = static::$key ?? '@id';

        return explode(':', $key);
    }

    /**
     * Create collection of data objects from api results
     *
     * @param Builder|null $builder
     * @return DataCollection
     */
    public static function filter(?Builder $builder = null): DataCollection
    {
        if (empty($builder)) {
            $builder = static::getDefaultFilter();
        }

        $cache_key = static::getFilterCacheKey(static::$apiObjectName, $builder);
        $results = cache()->remember($cache_key, now()->addSeconds(static::getCacheExpire()), function () use ($builder) {
            return @app(PaceConnector::class)->findObjects(
                object: static::getObjectName(),
                filter: $builder->toXPath()
            );
        });

        return static::collection(is_iterable($results) ? $results : []);
    }

    /**
     * @param string $prefix
     * @param Builder $builder
     * @return string
     */
    protected static function getFilterCacheKey(string $prefix, Builder $builder): string
    {
        $hash = Hash::make(serialize($builder));

        return "{$prefix}-{$hash}";
    }

    /**
     * @return int
     */
    protected static function getCacheExpire(): int
    {
        return static::$apiCacheFilterExpire ?? 90;
    }
}