<?php

namespace Ptb\Pace\Actions;

use Ptb\Pace\Actions\Estimates\AddContentFile;
use Ptb\Pace\Actions\Estimates\CreateEstimateInfo;
use Ptb\Pace\Actions\Estimates\CreateEstimatePartInfo;
use Ptb\Pace\Data\EstimatePart;
use Ptb\Pace\Enums\TaskStatus;
use Ptb\Pace\Models\PaceTask;
use Ptb\Pace\Models\PaceTaskPart;
use Ptb\Pace\Requests\AddEstimatePartRequest;
use Ptb\Pace\Requests\CreateEstimateRequest;
use Throwable;

final class ConvertTaskToEstimate
{
    public function __construct(
        private readonly CreateEstimateInfo     $estimateInfo,
        private readonly CreateEstimatePartInfo $estimatePartInfo,
        private readonly CreateEstimateRequest  $createEstimateRequest,
        private readonly AddEstimatePartRequest $addEstimatePartRequest,
        private readonly AddContentFile         $addContentFile,
    )
    {
    }

    /**
     * @param PaceTaskPart $part
     * @param EstimatePart $estimatePart
     * @return void
     */
    protected function handleContentFile(PaceTaskPart $part, EstimatePart $estimatePart): void
    {
        if (!empty($part->part_content_file_location)) {
            $this->addContentFile->handle(
                taskPart: $part,
                estimatePart: $estimatePart
            );
        }
    }

    /**
     * @param PaceTask $task
     * @param array $options
     * @return void
     */
    public function handle(
        PaceTask $task,
        array    $options = [],
    ): void
    {

        // Create the estimate
        try {
            foreach ($task->parts as $part_number => $part) {
                if ($part_number === 0) {
                    $estimate_info = $this->estimateInfo->handle(
                        task: $task,
                        options: [
                            'customer' => $part->automation->customer,
                        ]
                    );
                    $estimate_info->estimatePartInfo = $this->estimatePartInfo->handle(part: $part);

                    $estimate = $this->createEstimateRequest->handle(
                        estimateInfo: $estimate_info,
                    );

                    $estimate_parts = EstimatePart::filterByEstimate(
                        estimate: $estimate->id,
                    );

                    $estimate_part = $estimate_parts->first();

                    // update task with estimate number
                    $task->update([
                        'task_estimate_number' => $estimate->estimateNumber,
                        'task_status' => TaskStatus::EstimateCreated,
                    ]);
                } else {
                    $part_info = $this->estimatePartInfo->handle(part: $part);
                    $part_info->estimateID = (string)$estimate->id;
                    $estimate_part = $this->addEstimatePartRequest->handle(
                        estimatePartInfo: $part_info,
                    );
                }

                $this->handleContentFile(
                    part: $part,
                    estimatePart: $estimate_part
                );
            }

        } catch (Throwable $fault) {
            // most likely it will be a SoapFault exception that we are catching
            throw $fault;
        }
    }
}