<?php

namespace Ptb\Pace\Query;

use Closure;
use DateTime;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use InvalidArgumentException;
use JsonException;
use Ptb\Pace\PaceRestConnector;
use Ptb\Pace\Requests\FindObjects\FindAndSortObjects;
use Ptb\Pace\Requests\FindObjects\FindObjects;
use Ptb\Pace\Requests\FindObjects\FindSortAndLimitObjects;
use Saloon\Exceptions\Request\FatalRequestException;
use Saloon\Exceptions\Request\RequestException;
use Throwable;

class Builder
{
    /**
     * Valid operators.
     *
     * @var array
     */
    protected array $operators = ['=', '!=', '<', '>', '<=', '>='];

    /**
     * Valid functions.
     *
     * @var array
     */
    protected array $functions = ['contains', 'starts-with'];

    /**
     * The filters.
     *
     * @var array
     */
    protected array $filters = [];

    /**
     * The sorts.
     *
     * @var array
     */
    protected array $sorts = [];

    /**
     * @var string
     */
    protected string $objectName;

    /**
     * @var PaceRestConnector
     */
    protected PaceRestConnector $api;

    /**
     * @var string
     */
    protected string $dataClass;

    /**
     * Create a new instance.
     *
     * @param string $objectName
     * @param string $dataClass
     */
    public function __construct(string $objectName, string $dataClass)
    {
        $this->objectName = $objectName;

        $this->dataClass = $dataClass;

        $this->api = app(PaceRestConnector::class);
    }

    /**
     * Add a "contains" filter.
     *
     * @param string $xpath
     * @param mixed|null $value
     * @param string $boolean
     * @return self
     */
    public function contains(string $xpath, mixed $value = null, string $boolean = 'and'): Builder|static
    {
        return $this->filter($xpath, 'contains', $value, $boolean);
    }

    /**
     * Add an "or contains" filter.
     *
     * @param string $xpath
     * @param mixed|null $value
     * @return self
     */
    public function orContains(string $xpath, mixed $value = null): Builder|static
    {
        return $this->filter($xpath, 'contains', $value, 'or');
    }

    /**
     * Add a filter.
     *
     * @param string|Closure $xpath
     * @param string|null $operator
     * @param mixed|null $value
     * @param string $boolean
     * @return self
     */
    public function filter(string|Closure $xpath, string $operator = null, mixed $value = null, string $boolean = 'and'): Builder|static
    {
        if ($xpath instanceof Closure) {
            return $this->nestedFilter($xpath, $boolean);
        }

        if ($value === null && !$this->isOperator($operator)) {
            list($value, $operator) = [$operator, '='];
        }

        if (!$this->isOperator($operator) && !$this->isFunction($operator)) {
            throw new InvalidArgumentException("Operator '$operator' is not supported");
        }

        $this->filters[] = compact('xpath', 'operator', 'value', 'boolean');

        return $this;
    }

    /**
     * Add an "in" filter.
     *
     * @param string $xpath
     * @param array $values
     * @param string $boolean
     * @return self
     */
    public function in(string $xpath, array $values, string $boolean = 'and'): Builder|static
    {
        return $this->filter(function ($builder) use ($xpath, $values) {
            foreach ($values as $value) {
                $builder->filter($xpath, '=', $value, 'or');
            }
        }, null, null, $boolean);
    }

    /**
     * Add an "or in" filter.
     *
     * @param string $xpath
     * @param array $values
     * @return self
     */
    public function orIn(string $xpath, array $values): Builder|static
    {
        return $this->in($xpath, $values, 'or');
    }

    /**
     * Add a nested filter using a callback.
     *
     * @param Closure $callback
     * @param string $boolean
     * @return self
     */
    public function nestedFilter(Closure $callback, string $boolean = 'and'): static
    {
        $builder = new static($this->objectName, $this->dataClass);

        $callback($builder);

        $this->filters[] = compact('builder', 'boolean');

        return $this;
    }

    /**
     * Add an "or" filter.
     *
     * @param string $xpath
     * @param string|null $operator
     * @param mixed|null $value
     * @return self
     */
    public function orFilter(string $xpath, string $operator = null, mixed $value = null): static
    {
        return $this->filter($xpath, $operator, $value, 'or');
    }

    /**
     * Add a "starts-with" filter.
     *
     * @param string $xpath
     * @param mixed|null $value
     * @param string $boolean
     * @return self
     */
    public function startsWith(string $xpath, mixed $value = null, string $boolean = 'and'): static
    {
        return $this->filter($xpath, 'starts-with', $value, $boolean);
    }

    /**
     * Add an "or starts-with" filter.
     *
     * @param string $xpath
     * @param mixed|null $value
     * @return self
     */
    public function orStartsWith(string $xpath, mixed $value = null): static
    {
        return $this->filter($xpath, 'starts-with', $value, 'or');
    }

    /**
     * Add a sort.
     *
     * @param string $xpath
     * @param bool $descending
     * @return self
     */
    public function sort(string $xpath, bool $descending = false): static
    {
        $this->sorts[] = compact('xpath', 'descending');

        return $this;
    }

    /**
     * Get the XPath filter expression.
     *
     * @return string
     */
    public function toXPath(): string
    {
        $xpath = [];

        foreach ($this->filters as $filter) {

            if (isset($filter['builder'])) {
                $xpath[] = $this->compileNested($filter);

            } elseif ($this->isFunction($filter['operator'])) {
                $xpath[] = $this->compileFunction($filter);

            } else {
                $xpath[] = $this->compileFilter($filter);
            }
        }

        return $this->stripLeadingBoolean(implode(' ', $xpath));
    }

    /**
     * Get the XPath sort array.
     *
     * @return array|null
     */
    public function toXPathSort(): ?array
    {
        return Arr::wrap($this->sorts);
    }

    /**
     * Compile a simple filter.
     *
     * @param array $filter
     * @return string
     */
    protected function compileFilter(array $filter): string
    {
        return sprintf('%s %s %s %s',
            $filter['boolean'], $filter['xpath'], $filter['operator'], $this->value($filter['value']));
    }

    /**
     * Compile a function filter.
     *
     * @param array $filter
     * @return string
     */
    protected function compileFunction(array $filter): string
    {
        return sprintf('%s %s(%s, %s)',
            $filter['boolean'], $filter['operator'], $filter['xpath'], $this->value($filter['value']));
    }

    /**
     * Compile a nested filter.
     *
     * @param array $filter
     * @return string
     */
    protected function compileNested(array $filter): string
    {
        return sprintf('%s (%s)', $filter['boolean'], $filter['builder']->toXPath());
    }

    /**
     * Check if an operator is a valid function.
     *
     * @param string $operator
     * @return bool
     */
    protected function isFunction(string $operator): bool
    {
        return in_array($operator, $this->functions, true);
    }

    /**
     * Check if an operator is a valid operator.
     *
     * @param string $operator
     * @return bool
     */
    protected function isOperator(string $operator): bool
    {
        return in_array($operator, $this->operators, true);
    }

    /**
     * Strip the leading boolean from the expression.
     *
     * @param string $xpath
     * @return string
     */
    protected function stripLeadingBoolean(string $xpath): string
    {
        return preg_replace('/^and |^or /', '', $xpath);
    }

    /**
     * Get the XPath value for a PHP native type.
     *
     * @param mixed $value
     * @return string
     */
    protected function value(mixed $value): string
    {
        switch (true) {
            case ($value instanceof DateTime):
                return $this->date($value);

            case (is_int($value)):
            case (is_float($value)):
                return $value;

            case (is_bool($value)):
                return $value ? '\'true\'' : '\'false\'';

            default:
                return "\"$value\"";
        }
    }

    /**
     * Convert DateTime instance to XPath date function.
     *
     * @param DateTime $dt
     * @return string
     */
    protected function date(DateTime $dt): string
    {
        return $dt->format('\d\a\t\e(Y, n, j)');
    }

    /**
     * @return mixed
     * @throws FatalRequestException
     * @throws JsonException
     * @throws RequestException
     * @throws Throwable
     */
    public function get(): mixed
    {
        if (count($this->sorts)) {
            $request = new FindAndSortObjects(
                type: $this->objectName,
                xpath: $this->toXPath(),
                xpathSorts: $this->toXPathSort(),
            );
        } else {
            $request = new FindObjects;
            $request->query()->add('type', $this->objectName);
            $request->query()->add('xpath', $this->toXPath());
        }

        $response = $this->api->send($request);

        if ($response->failed()) {
            dump($response->headers());
            dump($response->body());
            $response->throw();
        };

        return call_user_func([$this->dataClass, 'collect'], $response->json(), Collection::class);
    }

    /**
     * @param int $page
     * @param int $limit
     * @return mixed
     * @throws FatalRequestException
     * @throws JsonException
     * @throws RequestException
     * @throws Throwable
     */
    public function paginate(int $page = 1, int $limit = 15): mixed
    {
        $request = new FindSortAndLimitObjects(
            type: $this->objectName,
            xpath: $this->toXPath(),
            offset: $page === 1 ? 1 : (($page - 1) * $limit) + 1,
            limit: $limit,
            xpathSorts: $this->toXPathSort(),
        );

        $response = $this->api->send($request);

        if ($response->failed()) {
                        dump($response->headers());
            dump($response->body());
            $response->throw();
        };

        return call_user_func([$this->dataClass, 'collect'], $response->json(), Collection::class);
    }

    /**
     * @return mixed
     * @throws FatalRequestException
     * @throws JsonException
     * @throws RequestException
     * @throws Throwable
     */
    public function first(): mixed
    {
        if (count($this->sorts)) {
            $request = new FindAndSortObjects(
                type: $this->objectName,
                xpath: $this->toXPath(),
                xpathSorts: $this->toXPathSort(),
            );
        } else {
            $request = new FindObjects;
            $request->query()->add('type', $this->objectName);
            $request->query()->add('xpath', $this->toXPath());
        }

        try {
            $response = $this->api->send($request);

            if ($response->failed()) {
                dump($response->headers());
                dump($response->body());
                $response->throw();
            }
        } catch (\Throwable $e) {
            dump($e);
        }

        return call_user_func([$this->dataClass, 'from'], $response->json(0));
    }
}
