<?php

namespace Ptb\Synapse\Resources;

use Ptb\Synapse\Events\SynapseOrderCreated;
use Ptb\Synapse\Events\SynapseOrderCreating;
use Ptb\Synapse\Requests\CreateOrder;
use Ptb\Synapse\Requests\GetOrderInfo;
use Ptb\Synapse\Requests\GetOrderLabels;
use Ptb\Synapse\SynapseConnector;
use ReflectionException;
use Saloon\Contracts\Response;
use Saloon\Exceptions\InvalidResponseClassException;
use Saloon\Exceptions\PendingRequestException;

final class OrderResource
{
    public function __construct(
        private readonly SynapseConnector $connector
    )
    {}

    /**
     * @param array $payload
     * @return Response
     * @throws InvalidResponseClassException
     * @throws PendingRequestException
     * @throws ReflectionException
     */
    public function create(array $payload): Response
    {
        event(
            new SynapseOrderCreating(
                payload: $payload
            )
        );

        $response = $this->connector->send(
            request: new CreateOrder(
                data: $payload
            )
        );

        event(
            new SynapseOrderCreated(
                payload: $payload,
                response: $response->json()
            )
        );

        return $response;
    }

    /**
     * @param int $orderId
     * @param int $shipId
     * @return Response
     * @throws InvalidResponseClassException
     * @throws PendingRequestException
     * @throws ReflectionException
     */
    public function info(int $orderId, int $shipId = 1): Response
    {
        return $this->connector->send(
            request: new GetOrderInfo(
                orderId: $orderId,
                shipId: $shipId
            )
        );
    }

    /**
     * @param int $orderId
     * @param int $shipId
     * @return Response
     * @throws ReflectionException
     * @throws InvalidResponseClassException
     * @throws PendingRequestException
     */
    public function label(int $orderId, int $shipId = 1): Response
    {
        return $this->connector->send(
            request: new GetOrderLabels(
                orderId: $orderId,
                shipId: $shipId
            )
        );
    }

}