<?php

namespace Ptb\Saml2;

use InvalidArgumentException;
use OneLogin\Saml2\Auth;
use OneLogin\Saml2\Error;
use OneLogin\Saml2\Utils;
use OneLogin\Saml2\ValidationError;
use Ptb\Saml2\Events\Saml2LogoutEvent;

class Saml2 extends Auth
{
    private mixed $config;

    public function __construct()
    {
        Utils::setProxyVars(true);

        $this->config = config('saml2');
        if (is_null($this->config)) {
            throw new InvalidArgumentException('SAML 2.0 is not Configured.');
        }

        if (empty($this->config['sp']['entityId'])) {
            $this->config['sp']['entityId'] = URL::route('auth.saml2.show');
        }

        $acs_route = data_get($this->config, 'sp.assertionConsumerService.url');
        $this->config['sp']['assertionConsumerService']['url'] = route($acs_route);

        $sls_route = data_get($this->config, 'sp.singleLogoutService.url');
        $this->config['sp']['singleLogoutService']['url'] = route($sls_route);

        parent::__construct($this->config);
    }

    /**
     * @return array|null
     * @throws Error
     * @throws ValidationError
     */
    public function acs(): ?array
    {
        $this->processResponse();
        $errors = $this->getErrors();

        if (!empty($errors)) {
            return array('error' => $errors, 'last_error_reason' => $this->getLastErrorReason());
        }

        if (!$this->isAuthenticated()) {
            return array('error' => 'Could not authenticate', 'last_error_reason' => $this->getLastErrorReason());
        }

        return null;
    }

    /**
     * @param bool $retrieveParametersFromServer
     * @return array|null
     * @throws Error
     */
    public function sls(bool $retrieveParametersFromServer = false): ?array
    {
        $session_callback = fn() => event(new Saml2LogoutEvent());
        $this->processSLO(false, null, $retrieveParametersFromServer, $session_callback);
        $errors = $this->getErrors();

        if (!empty($errors)) {
            return array('error' => $errors, 'last_error_reason' => $this->getLastErrorReason());
         }

        return null;
    }

    /**
     * @return string
     * @throws Error
     */
    public function getMetadata(): string
    {
        $settings = $this->getSettings();
        $metadata = $settings->getSPMetadata();
        $errors = $settings->validateMetadata($metadata);

        if (empty($errors)) {
            return $metadata;
        } else {
            throw new InvalidArgumentException(
                'Invalid SP metadata: ' . implode(', ', $errors),
                Error::METADATA_SP_INVALID
            );
        }
    }

    /**
     * @return string
     */
    public function getUserId(): string
    {
        return $this->getNameId();
    }

    /**
     * @return string|null
     */
    public function getIntendedUrl(): ?string
    {
        $relayState = app('request')->input('RelayState');
        if ($relayState && url()->full() != $relayState) {
            return $relayState;
        }

        return null;
    }
}