<?php

namespace Ptb\Saml2\Http\Controllers;

use Illuminate\Http\Request;
use OneLogin\Saml2\Error;
use OneLogin\Saml2\ValidationError;
use Ptb\Saml2\Events\Saml2LoginEvent;
use Ptb\Saml2\Events\Saml2LogoutEvent;
use Ptb\Saml2\Saml2;

class Saml2Controller
{
    /**
     * @param Saml2 $saml2
     * @throws Error
     */
    public function index(Saml2 $saml2)
    {
        $saml2->login(
            returnTo: route(config('saml2.loginRoute'))
        );
    }

    /**
     * @throws Error
     */
    public function show(Saml2 $saml2)
    {
        return response(
            content: $saml2->getMetadata(),
            status: 200,
            headers: [
                'Content-Type' => 'text/xml',
            ]
        );
    }

    /**
     * @throws Error
     * @throws ValidationError
     */
    public function store(Saml2 $saml2)
    {
        if ($errors = $saml2->acs()) {
            if (count($errors) > 0) {
                logger()->error(
                    message: 'Saml2 ACS Error Reason ',
                    context: [
                        'error' => $saml2->getLastErrorReason()
                    ]
                );

                logger()->error(
                    message: 'SAML2 ACS Error(s)',
                    context: $errors
                );

                return redirect(config('saml2.errorRoute'));
            }
        }

        event(new Saml2LoginEvent($saml2));

        if ($redirectUrl = $saml2->getIntendedUrl()) {
            return redirect($redirectUrl);
        } else {
            return redirect('/');
        }
    }

    /**
     * @return void
     */
    public function destroy(Saml2 $saml2, Request $request)
    {
        event(new Saml2LogoutEvent());

        try {
            $returnTo = $request->get('returnTo', route(config('saml2.logoutRoute')));
            $sessionIndex = $request->get('sessionIndex');
            $nameId = $request->get('nameId');
            $saml2->logout($returnTo, [], $nameId, $sessionIndex);
        } catch (Error $error) {
            // do nothing
        }
    }

    /**
     * @throws Error
     */
    public function forceDelete(Saml2 $saml2)
    {
        $errors = $saml2->sls(config( 'auth.saml2.retrieveParametersFromServer'));
        if (!empty($errors)) {
            logger()->error('SAML2 SLS Error(s)', $errors);
        }

        return redirect(route(config('saml2.logoutRoute')));
    }
}
