<?php

namespace Ptb\OrderDesk\Transformers;

use Exception;

class OrderItemTransformer
{
    protected ?string $name;
    protected float $price;
    protected int $quantity;
    protected ?float $weight;
    protected string $code;
    protected string $deliveryType;
    protected ?string $categoryCode;
    protected ?array $variationList;
    protected ?array $metadata;

    public function __construct()
    {
        $this->name = null;
        $this->price = 0.00;
        $this->quantity = 1;
        $this->weight = null;
        $this->code = '';
        $this->deliveryType = 'ship';
        $this->categoryCode = null;
        $this->variationList = null;
        $this->metadata = null;
    }


    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @return float
     */
    public function getPrice(): float
    {
        return $this->price;
    }

    /**
     * @return int
     */
    public function getQuantity(): int
    {
        return $this->quantity;
    }

    /**
     * @return float|null
     */
    public function getWeight(): ?float
    {
        return $this->weight;
    }

    /**
     * @return string
     */
    public function getCode(): string
    {
        return $this->code;
    }

    /**
     * @return string
     */
    public function getDeliveryType(): string
    {
        return $this->deliveryType;
    }

    /**
     * @return string|null
     */
    public function getCategoryCode(): ?string
    {
        return $this->categoryCode;
    }

    /**
     * @return array|null
     */
    public function getVariationList(): ?array
    {
        return $this->variationList;
    }

    /**
     * @return array|null
     */
    public function getMetadata(): ?array
    {
        return $this->metadata;
    }

    /**
     * @param string|null $name
     * @return OrderItemTransformer
     */
    public function setName(?string $name): OrderItemTransformer
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @param float $price
     * @return OrderItemTransformer
     */
    public function setPrice(float $price): OrderItemTransformer
    {
        $this->price = $price;
        return $this;
    }

    /**
     * @param int $quantity
     * @return OrderItemTransformer
     */
    public function setQuantity(int $quantity): OrderItemTransformer
    {
        $this->quantity = $quantity;
        return $this;
    }

    /**
     * @param float|null $weight
     * @return OrderItemTransformer
     */
    public function setWeight(?float $weight): OrderItemTransformer
    {
        $this->weight = $weight;
        return $this;
    }

    /**
     * @param string $code
     * @return OrderItemTransformer
     */
    public function setCode(string $code): OrderItemTransformer
    {
        $this->code = $code;
        return $this;
    }

    /**
     * @param string $deliveryType
     * @return OrderItemTransformer
     * @throws Exception
     */
    public function setDeliveryType(string $deliveryType): OrderItemTransformer
    {
        if (! in_array($deliveryType, ['ship', 'noship', 'download', 'future'])) {
            throw new Exception('Provided delivery type not one of ship,noship,download,future');
        }

        $this->deliveryType = $deliveryType;
        return $this;
    }

    /**
     * @param string|null $categoryCode
     * @return OrderItemTransformer
     */
    public function setCategoryCode(?string $categoryCode): OrderItemTransformer
    {
        $this->categoryCode = $categoryCode;
        return $this;
    }

    /**
     * @param array|null $variationList
     * @return OrderItemTransformer
     */
    public function setVariationList(?array $variationList): OrderItemTransformer
    {
        $this->variationList = $variationList;
        return $this;
    }

    /**
     * @param array|null $metadata
     * @return OrderItemTransformer
     */
    public function setMetadata(?array $metadata): OrderItemTransformer
    {
        $this->metadata = $metadata;
        return $this;
    }

    /**
     * @return array
     */
    public function toArray(): array
    {
        return array_filter([
            'name' => $this->name,
            'price' => $this->price,
            'quantity' => $this->quantity,
            'weight' => $this->weight,
            'code' => $this->code,
            'delivery_type' => $this->deliveryType,
            'category_code' => $this->categoryCode,
            'variation_list' => $this->variationList,
            'metadata' => $this->metadata
        ]);
    }
}