<?php

namespace Ptb\CartRover\Models;


use Illuminate\Contracts\Support\Jsonable;
use Ptb\CartRover\Contracts\CanHydrateFromArray;
use function data_get;

class Order implements Jsonable, CanHydrateFromArray
{
    public const StatusNew = 'new';
    public const StatusAtWms = 'at_wms';
    public const StatusNewOrAtWms = 'new_or_at_wms';
    public const StatusPartial = 'partial';
    public const StatusShipped = 'shipped';
    public const StatusConfirmed = 'confirmed';
    public const StatusShippedOrConfirmed = 'shipped_or_confirmed';
    public const StatusError = 'error';
    public const StatusCanceled = 'canceled';
    public const StatusAny = 'any';

    public ?string $action_dt = null;
    public ?string $ani_phone = null;
    public ?float $balance = null;
    public ?string $black_list = null;
    public ?string $call_queue = null;
    public ?string $carrier = null;
    public ?string $cass_code_cust = null;
    public ?string $cass_code_ship = null;
    public ?string $cass_date = null;
    public ?string $cass_error_cust = null;
    public ?string $cass_error_ship = null;
    public ?int $cc_last_four = null;
    public ?int $check_account_no = null;
    public ?string $check_type = null;
    public ?int $check_no = null;
    public ?string $check_bank_id = null;
    public ?string $check_cust_bank = null;
    public ?string $check_cust_id_num = null;
    public ?string $check_cust_id_state = null;
    public ?int $check_cust_id_mm = null;
    public ?int $check_cust_id_dd = null;
    public ?int $check_cust_id_yy = null;
    public ?string $check_cust_id_type = null;
    public ?string $clerk_disposition = null;
    public ?string $clerk_disp_dt = null;
    public ?string $credit_card_no = null;
    public ?string $credit_error_code = null;
    public ?string $credit_score = null;

    /**
     * @var ?string
     * ISO 4217 Currency Code - Most systems will assume a default of the local currency
     */
    public ?string $currency_code = null;
    public ?string $cust_address_1 = null;
    public ?string $cust_address_2 = null;
    public ?string $cust_address_3 = null;
    public ?string $cust_city = null;

    /**
     * @var ?string
     * Required if cust_first_name or cust_last_name not set
     */
    public ?string $cust_company = null;
    public ?string $cust_country = null;
    public ?string $cust_e_mail = null;

    /**
     * @var string
     * Required if cust_company or cust_last_name not set
     */
    public ?string $cust_first_name = null;

    /**
     * @var string
     * Required if cust_company or cust_first_name not set
     */
    public ?string $cust_last_name = null;
    public ?string $cust_phone = null;

    /**
     * @var string
     * Optional additional unique order number.
     * Order will reject if value is not unique.
     * This can generally be set to the same value as cust_ref, but supports longer order numbers.
     */
    public ?string $cust_po_no = null;

    /**
     * @var string
     * Unique Order Number. Always check the cust_ref field field in the response!
     * There are multiple reasons CartRover may accept the order, but change the cust_ref value.
     * The most common reasons are:
     * 1) The cust_ref sent was over 16 chars in length and had to be shortened.
     * 2) The cust_ref was already used by a different order source for this merchant and has been modified to ensure uniqueness.
     */
    public string $cust_ref;
    public ?string $cust_state = null;
    public ?string $cust_title = null;
    public ?string $cust_zip = null;
    public ?string $custom_field_1 = null;
    public ?string $custom_field_2 = null;
    public ?string $custom_field_3 = null;
    public ?string $custom_field_4 = null;
    public ?string $custom_field_5 = null;
    public ?string $customer_id = null;
    public ?int $cvv = null;
    public ?bool $duplicate_sw = null;
    public ?string $expiration_date = null;
    public ?string $expected_delivery_date = null;
    public ?array $extra_fields = null;
    public ?string $fail_action = null;
    public ?string $filename = null;
    public ?bool $first_attempt = null;
    public ?string $fraud_score = null;
    public ?bool $gift_card_sw = null;
    public ?string $gift_message = null;
    public ?bool $gift_wrap = null;
    public ?float $grand_total = null;
    public ?string $ifraud_error_code = null;
    public ?string $installment_program = null;

    /**
     * @var OrderItem[]
     */
    public array $items;
    public ?string $location = null;
    public ?bool $load_override_sw = null;
    public ?int $media_week = null;
    public ?string $ncoa_code_ship = null;
    public ?string $ncoa_code_cust = null;
    public ?string $ncoa_date = null;

    /**
     * @var string
     * Format: YYYY-MM-DD Defaults to current date if not provided.
     */
    public string $order_date;

    /**
     * @var string
     * Warning: This will override any Order Source value defined in CartRover
     */
    public ?string $order_source = null;
    public ?float $order_discount = null;
    public ?string $orig_file_no  = null;
    public ?string $pay_type  = null;
    public ?string $pre_auth_code = null;
    public ?float $pre_auth_amt = null;
    public ?string $pre_auth_id = null;
    public ?string $prepaid_order = null;
    public ?string $promo_code = null;
    public ?string $rep_disposition = null;
    public ?string $rep_disp_dt = null;
    public ?string $regional_center = null;
    public ?string $regional_order_no = null;
    public ?string $regional_ship_date = null;
    public ?string $requested_ship_date = null;
    public ?string $resubmit_date = null;

    /**
     * @var bool
     * Request warehouses system to perform additional routing if supported
     */
    public ?bool $routing_sw = null;

    /**
     * @var ?float
     * If tax is set on the order line then this should match the sum of all order line taxes.
     */
    public ?float $sales_tax = null;
    public ?string $ship_account_no = null;
    public string $ship_address_1;
    public ?string $ship_address_2 = null;
    public ?string $ship_address_3 = null;

    /**
     * @var ?string
     * R - Residential, C - Commercial
     */
    public ?string $ship_address_type = null;
    public string $ship_city;

    /**
     * @var ?string
     * Shipping Method to be used by the warehouse
     */
    public ?string $ship_code = null;
    public ?string $ship_code_description = null;

    /**
     * @var ?string
     * Required if ship_first_name or ship_last_name not set
     */
    public ?string $ship_company = null;

    /**
     * @var string
     * ISO 2 or 3 character code preferred.
     * If full name is passed, best effort will be made to convert it to an ISO code.
     */
    public string $ship_country;
    public ?string $ship_e_mail = null;

    /**
     * @var ?string
     * Required if ship_company or ship_last_name not set
     */
    public ?string $ship_first_name = null;

    /**
     * @var bool
     * If true, billing address will be set to the same as the shipping address.
     * Any values passed for billing address will be ignored.
     */
    public ?bool $ship_is_billing = null;

    /**
     * @var ?string
     * Required if ship_company or ship_first_name not set
     */
    public ?string $ship_last_name = null;

    /**
     * @var string
     *    ISO 2 or 3 character code preferred. If full name is passed, best effort will be made to convert it to an ISO code.
     */
    public string $ship_state;
    public ?string $ship_phone = null;
    public ?string $ship_title = null;
    public string $ship_zip;

    /**
     * @var ?float
     * The total shipping charge for the order.
     * This has no relation to the order line shipping_surcharge field.
     */
    public ?float $shipping_handling = null;
    public ?string $shipping_instructions = null;
    public ?string $special_services = null;
    public ?float $sub_total = null;
    public ?float $tax_exempt_sw = null;

    /**
     * @var ?bool
     * Y - means credit_card_no is provided and is already tokenized.
     * If passing Credit Card data, this must by Y.
     * CartRover does not allow untokenized credit cards.
     */
    public ?bool $token_sw = null;
    public ?string $vendor_phone = null;
    public ?float $weight = null;
    public ?string $xfraud_error_code = null;

    /**
     * @param array $attributes
     * @return Order
     */
    public static function make(array $attributes = []): Order
    {
        $vars = get_class_vars(self::class);
        $order = new self;
        foreach ($vars as $var => $ignore) {
            $order->$var = match ($var) {
                'items' => call_user_func(function () use ($attributes, $var) {
                    $order_items = [];
                    $items = data_get($attributes, $var);
                    if (is_iterable($items)) {
                        foreach ($items as $item) {
                            $order_items[] = OrderItem::make($item);
                        }
                    }
                    return $order_items;
                }),
                'cust_ref', 'order_source', 'ship_address_1', 'ship_city', 'ship_state', 'ship_zip' => data_get($attributes, $var, ''),
                'order_date' => data_get($attributes, $var, now()->format('Y-m-d')),
                default => data_get($attributes, $var)
            };
        }

        return $order;
    }

    /**
     * @param int $options
     * @return array
     */
    public function toJson($options = 0): array
    {
        $data = [];
        $vars = get_class_vars(self::class);
        foreach ($vars as $var => $ignore) {
            if (!is_null($this->$var)) {
                $data[$var] = match ($var) {
                    'items' => call_user_func(function () {
                        $item_data = [];
                        foreach ($this->items as $item) {
                            $item_data[] = $item->toJson();
                        }
                        return $item_data;
                    }),
                    default => $this->$var ?? null
                };
            }
        }

        return $data;
    }
}
