<?php

namespace Ptb\CartRover\Cart\Models;


use Illuminate\Contracts\Support\Jsonable;
use Ptb\CartRover\Cart\Contracts\CanHydrateFromArray;
use function data_get;

class Order implements Jsonable, CanHydrateFromArray
{
    public const StatusNew = 'new';
    public const StatusAtWms = 'at_wms';
    public const StatusNewOrAtWms = 'new_or_at_wms';
    public const StatusPartial = 'partial';
    public const StatusShipped = 'shipped';
    public const StatusConfirmed = 'confirmed';
    public const StatusShippedOrConfirmed = 'shipped_or_confirmed';
    public const StatusError = 'error';
    public const StatusCanceled = 'canceled';
    public const StatusAny = 'any';

    public ?string $action_dt;
    public ?string $ani_phone;
    public ?float $balance;
    public ?string $black_list;
    public ?string $call_queue;
    public ?string $carrier;
    public ?string $cass_code_cust;
    public ?string $cass_code_ship;
    public ?string $cass_date;
    public ?string $cass_error_cust;
    public ?string $cass_error_ship;
    public ?int $cc_last_four;
    public ?int $check_account_no;
    public ?string $check_type;
    public ?int $check_no;
    public ?string $check_bank_id;
    public ?string $check_cust_bank;
    public ?string $check_cust_id_num;
    public ?string $check_cust_id_state;
    public ?int $check_cust_id_mm;
    public ?int $check_cust_id_dd;
    public ?int $check_cust_id_yy;
    public ?string $check_cust_id_type;
    public ?string $clerk_disposition;
    public ?string $clerk_disp_dt;
    public ?string $credit_card_no;
    public ?string $credit_error_code;
    public ?string $credit_score;

    /**
     * @var ?string
     * ISO 4217 Currency Code - Most systems will assume a default of the local currency
     */
    public ?string $currency_code;
    public ?string $cust_address_1;
    public ?string $cust_address_2;
    public ?string $cust_address_3;
    public ?string $cust_city;

    /**
     * @var ?string
     * Required if cust_first_name or cust_last_name not set
     */
    public ?string $cust_company;
    public ?string $cust_country;
    public ?string $cust_e_mail;

    /**
     * @var string
     * Required if cust_company or cust_last_name not set
     */
    public ?string $cust_first_name;

    /**
     * @var string
     * Required if cust_company or cust_first_name not set
     */
    public ?string $cust_last_name;
    public ?string $cust_phone;

    /**
     * @var string
     * Optional additional unique order number.
     * Order will reject if value is not unique.
     * This can generally be set to the same value as cust_ref, but supports longer order numbers.
     */
    public ?string $cust_po_no;

    /**
     * @var string
     * Unique Order Number. Always check the cust_ref field field in the response!
     * There are multiple reasons CartRover may accept the order, but change the cust_ref value.
     * The most common reasons are:
     * 1) The cust_ref sent was over 16 chars in length and had to be shortened.
     * 2) The cust_ref was already used by a different order source for this merchant and has been modified to ensure uniqueness.
     */
    public string $cust_ref;
    public ?string $cust_state;
    public ?string $cust_title;
    public ?string $cust_zip;
    public ?string $custom_field_1;
    public ?string $custom_field_2;
    public ?string $custom_field_3;
    public ?string $custom_field_4;
    public ?string $custom_field_5;
    public ?string $customer_id;
    public ?int $cvv;
    public ?bool $duplicate_sw;
    public ?string $expiration_date;
    public ?string $expected_delivery_date;
    public ?array $extra_fields;
    public ?string $fail_action;
    public ?string $filename;
    public ?bool $first_attempt;
    public ?string $fraud_score;
    public ?bool $gift_card_sw;
    public ?string $gift_message;
    public ?bool $gift_wrap;
    public ?float $grand_total;
    public ?string $ifraud_error_code;
    public ?string $installment_program;

    /**
     * @var OrderItem[]
     */
    public array $items;
    public ?string $location;
    public ?bool $load_override_sw;
    public ?int $media_week;
    public ?string $ncoa_code_ship;
    public ?string $ncoa_code_cust;
    public ?string $ncoa_date;

    /**
     * @var string
     * Format: YYYY-MM-DD Defaults to current date if not provided.
     */
    public string $order_date;

    /**
     * @var string
     * Warning: This will override any Order Source value defined in CartRover
     */
    public string $order_source;
    public ?float $order_discount;
    public ?string $orig_file_no;
    public ?string $pay_type;
    public ?string $pre_auth_code;
    public ?float $pre_auth_amt;
    public ?string $pre_auth_id;
    public ?string $prepaid_order;
    public ?string $promo_code;
    public ?string $rep_disposition;
    public ?string $rep_disp_dt;
    public ?string $regional_center;
    public ?string $regional_order_no;
    public ?string $regional_ship_date;
    public ?string $requested_ship_date;
    public ?string $resubmit_date;

    /**
     * @var bool
     * Request warehouses system to perform additional routing if supported
     */
    public ?bool $routing_sw;

    /**
     * @var ?float
     * If tax is set on the order line then this should match the sum of all order line taxes.
     */
    public ?float $sales_tax;
    public ?string $ship_account_no;
    public string $ship_address_1;
    public ?string $ship_address_2;
    public ?string $ship_address_3;

    /**
     * @var ?string
     * R - Residential, C - Commercial
     */
    public ?string $ship_address_type;
    public string $ship_city;

    /**
     * @var ?string
     * Shipping Method to be used by the warehouse
     */
    public ?string $ship_code;
    public ?string $ship_code_description;

    /**
     * @var ?string
     * Required if ship_first_name or ship_last_name not set
     */
    public ?string $ship_company;

    /**
     * @var string
     * ISO 2 or 3 character code preferred.
     * If full name is passed, best effort will be made to convert it to an ISO code.
     */
    public string $ship_country;
    public ?string $ship_e_mail;

    /**
     * @var ?string
     * Required if ship_company or ship_last_name not set
     */
    public ?string $ship_first_name;

    /**
     * @var bool
     * If true, billing address will be set to the same as the shipping address.
     * Any values passed for billing address will be ignored.
     */
    public ?bool $ship_is_billing;

    /**
     * @var ?string
     * Required if ship_company or ship_first_name not set
     */
    public ?string $ship_last_name;

    /**
     * @var string
     *    ISO 2 or 3 character code preferred. If full name is passed, best effort will be made to convert it to an ISO code.
     */
    public string $ship_state;
    public ?string $ship_phone;
    public ?string $ship_title;
    public string $ship_zip;

    /**
     * @var ?float
     * The total shipping charge for the order.
     * This has no relation to the order line shipping_surcharge field.
     */
    public ?float $shipping_handling;
    public ?string $shipping_instructions;
    public ?string $special_services;
    public ?float $sub_total;
    public ?float $tax_exempt_sw;

    /**
     * @var ?bool
     * Y - means credit_card_no is provided and is already tokenized.
     * If passing Credit Card data, this must by Y.
     * CartRover does not allow untokenized credit cards.
     */
    public ?bool $token_sw;
    public ?string $vendor_phone;
    public ?float $weight;
    public ?string $xfraud_error_code;

    /**
     * @param array $attributes
     * @return Order
     */
    public static function make(array $attributes = []): Order
    {
        $vars = get_class_vars(self::class);
        $order = new self;
        foreach ($vars as $var => $ignore) {
            $order->$var = match ($var) {
                'items' => call_user_func(function () use ($attributes, $var) {
                    $order_items = [];
                    $items = data_get($attributes, $var);
                    if (is_iterable($items)) {
                        foreach ($items as $item) {
                            $order_items[] = OrderItem::make($item);
                        }
                    }
                    return $order_items;
                }),
                'cust_ref', 'order_source', 'ship_address_1', 'ship_city', 'ship_state', 'ship_zip' => data_get($attributes, $var, ''),
                'order_date' => data_get($attributes, $var, now()->format('Y-m-d')),
                default => data_get($attributes, $var)
            };
        }

        return $order;
    }

    /**
     * @param int $options
     * @return array
     */
    public function toJson($options = 0): array
    {
        $data = [];
        $vars = get_class_vars(self::class);
        foreach ($vars as $var => $ignore) {
            if (!is_null($this->$var)) {
                $data[$var] = match ($var) {
                    'items' => call_user_func(function () {
                        $item_data = [];
                        foreach ($this->items as $item) {
                            $item_data[] = $item->toJson();
                        }
                        return $item_data;
                    }),
                    default => $this->$var
                };
            }
        }

        return $data;
    }
}
