<?php

namespace Ptb\Grafx\Requests;

use DateTime;
use Saloon\Enums\Method;
use Saloon\Http\Request;

/**
 * Represents a request to read templates from a specific template collection within an environment.
 *
 * Extends the `Request` class and is configured to use the GET HTTP method.
 * It allows specifying optional query parameters like next page token and limit for pagination.
 *
 * The endpoint is determined dynamically based on the provided environment and template collection ID.
 */
class ReadTemplateCollectionTemplatesRequest extends Request
{
	protected Method $method = Method::GET;


	public function resolveEndpoint(): string
	{
		return "/api/v1/environment/$this->environment/template-collections/$this->templateCollectionId/templates";
	}


	/**
	 * @param string $environment Name of the environment
	 * @param string $templateCollectionId ID of the template collection
	 * @param null|string $nextPageToken Next page token
	 * @param null|int $limit Amount of resources per response
	 */
	public function __construct(
		protected string $environment,
		protected string $templateCollectionId,
		protected ?string $nextPageToken = null,
		protected ?int $limit = null,
	) {
	}


	public function defaultQuery(): array
	{
		return array_filter(['nextPageToken' => $this->nextPageToken, 'limit' => $this->limit]);
	}
}
