<?php

namespace Ptb\Grafx;

use Ptb\Grafx\Requests\ReadPingRequest;
use Ptb\Grafx\Requests\ReadServerInfoRequest;
use Ptb\Grafx\Resource\OutputSettingResource;
use Ptb\Grafx\Resource\ProjectDocumentResource;
use Ptb\Grafx\Resource\ProjectDocumentUrlResource;
use Ptb\Grafx\Resource\ProjectPreviewResource;
use Ptb\Grafx\Resource\ProjectPreviewStatusResource;
use Ptb\Grafx\Resource\ProjectResource;
use Ptb\Grafx\Resource\TaskOutputErrorResource;
use Ptb\Grafx\Resource\TaskOutputResource;
use Ptb\Grafx\Resource\TaskResource;
use Ptb\Grafx\Resource\TemplateCollectionResource;
use Ptb\Grafx\Resource\TemplateDownloadResource;
use Ptb\Grafx\Resource\TemplatePreviewResource;
use Ptb\Grafx\Resource\TemplatePreviewStatusResource;
use Ptb\Grafx\Resource\TemplateResource;
use Saloon\Exceptions\Request\FatalRequestException;
use Saloon\Exceptions\Request\RequestException;
use Saloon\Helpers\OAuth2\OAuthConfig;
use Saloon\Http\Connector;
use Saloon\Http\OAuth2\GetClientCredentialsTokenRequest;
use Saloon\Http\Response;
use Saloon\Traits\OAuth2\ClientCredentialsGrant;
use Saloon\Traits\Plugins\AcceptsJson;

/**
 * Class GrafxConnector
 *
 * This class extends the Connector class and provides functionalities to interact with
 * the Grafx API, including authentication and operations related to various entities
 * such as projects, templates, and output settings.
 */
class GrafxConnector extends Connector
{
    use AcceptsJson, ClientCredentialsGrant;

    /**
     * @return string
     */
    public function resolveStudioBaseUrl(): string
    {
        return $this->resolveBaseUrl() . '/api/v1/environment/' . $this->environment;
    }

    /**
     * @return string
     */
	public function resolveBaseUrl(): string
	{
        return "https://$this->environment.chili-publish.online/grafx";
	}

    /**
     * @return OAuthConfig
     */
    protected function defaultOauthConfig(): OAuthConfig
    {
        return OAuthConfig::make()
            ->setClientId(config('services.grafx.clientId'))
            ->setClientSecret(config('services.grafx.clientSecret'))
	        ->setDefaultScopes(['*'])
            ->setTokenEndpoint('https://integration-login.chiligrafx.com/oauth/token')
            ->setRequestModifier(function (GetClientCredentialsTokenRequest $request) {
                $request->body()->add('audience', 'https://chiligrafx.com');
            });
    }

    public function __construct(
        public string $environment
    )
    {}

    /**
     * @return Response
     * @throws FatalRequestException
     * @throws RequestException
     */
    public function ping(): Response
    {
        return $this->send(
            request: new ReadPingRequest()
        );
    }

    /**
     * @return Response
     * @throws FatalRequestException
     * @throws RequestException
     */
    public function serverInfo(): Response
    {
        return $this->send(
            request: new ReadServerInfoRequest()
        );
    }

    /**
     * @return OutputSettingResource
     */
    public function outputSetting(): OutputSettingResource
    {
        return new OutputSettingResource($this);
    }

    /**
     * @return ProjectDocumentUrlResource
     */
    public function projectDocumentUrl(): ProjectDocumentUrlResource
    {
        return new ProjectDocumentUrlResource($this);
    }

    /**
     * @return ProjectDocumentResource
     */
    public function projectDocument(): ProjectDocumentResource
    {
        return new ProjectDocumentResource($this);
    }

    /**
     * @return ProjectPreviewResource
     */
    public function projectPreview(): ProjectPreviewResource
    {
        return new ProjectPreviewResource($this);
    }

    /**
     * @return ProjectPreviewStatusResource
     */
    public function projectPreviewStatus(): ProjectPreviewStatusResource
    {
        return new ProjectPreviewStatusResource($this);
    }

    /**
     * @return ProjectResource
     */
    public function project(): ProjectResource
	{
        return new ProjectResource($this);
	}

    /**
     * @return TaskOutputErrorResource
     */
    public function taskOutputError(): TaskOutputErrorResource
    {
        return new TaskOutputErrorResource($this);
    }

    /**
     * @return TaskOutputResource
     */
    public function taskOutput(): TaskOutputResource
	{
        return new TaskOutputResource($this);
	}

    /**
     * @return TaskResource
     */
    public function task(): TaskResource
    {
        return new TaskResource($this);
    }

    /**
     * @return TemplateDownloadResource
     */
    public function templateDownload(): TemplateDownloadResource
    {
        return new TemplateDownloadResource($this);
    }

    /**
     * @return TemplatePreviewResource
     */
    public function templatePreview(): TemplatePreviewResource
    {
        return new TemplatePreviewResource($this);
    }

    /**
     * @return TemplateResource
     */
    public function templatePreviewStatus(): TemplatePreviewStatusResource
    {
        return new TemplatePreviewStatusResource($this);
    }

    /**
     * @return TemplateResource
     */
    public function template(): TemplateResource
    {
        return new TemplateResource($this);
    }

    /**
     * @return TemplateCollectionResource
     */
    public function templateCollection(): TemplateCollectionResource
    {
        return new TemplateCollectionResource($this);
    }
}
