<?php

namespace Ptb\Grafx\Resource;

use Ptb\Grafx\Requests\Templates\CreateTemplate;
use Ptb\Grafx\Requests\Templates\DeleteTemplate;
use Ptb\Grafx\Requests\Templates\DownloadTemplate;
use Ptb\Grafx\Requests\Templates\GetTemplates;
use Ptb\Grafx\Requests\Templates\ReturnsAllTemplatesInTheEnvironmentIncludingSubfolders;
use Ptb\Grafx\Requests\Templates\GetTemplatePreview;
use Ptb\Grafx\Requests\Templates\ReturnsPreviewGenerationTaskStatus;
use Ptb\Grafx\Requests\Templates\GetTemplate;
use Ptb\Grafx\Requests\Templates\ReturnsTheTemplateCollectionsThatContainTheTemplate;
use Ptb\Grafx\Requests\Templates\StoresTemplateInTemplateCollections;
use Ptb\Grafx\Requests\Templates\UpdateTemplate;
use Ptb\Grafx\Resource;
use Saloon\Http\Response;

class Templates extends Resource
{

	/**
	 * @param string $environment Name of the environment
	 * @param string $name Name of template
	 * @param string $folderPath Folder to place created template (NOTE: forward slashes only, should start with a slash)
	 */
	public function createTemplate(string $environment, string $name, string $folderPath): Response
	{
		return $this->connector->send(new CreateTemplate($environment, $name, $folderPath));
	}


	/**
	 * @param string $environment Name of the environment
	 * @param string $nextPageToken Next page token
	 * @param int $limit Amount of items (resources and folders) per response
	 * @param string $sortBy Name of the field by which the sorting is performed
	 * @param mixed $sortOrder Order of resources, ascending or descending
	 * @param string $folder Folder in which search should be done (NOTE: forward slashes only, should start with a slash)
	 * @param string $search Search by name, relativePath or resource ID
	 */
	public function findAll(
		string $environment,
		?string $nextPageToken = null,
		?int $limit = null,
		?string $sortBy = null,
		mixed $sortOrder = null,
		?string $folder = null,
		?string $search = null,
	): Response
	{
		return $this->connector->send(new GetTemplates($environment, $nextPageToken, $limit, $sortBy, $sortOrder, $folder, $search));
	}


	/**
	 * @param string $environment Name of the environment
	 * @param string $templateId ID of the template
	 */
	public function getTemplate(string $environment, string $templateId): Response
	{
		return $this->connector->send(new GetTemplate($environment, $templateId));
	}


	/**
	 * @param string $environment Name of the environment
	 * @param string $templateId ID of the template
	 * @param string $name Name of the template
	 * @param string $folderPath Folder to move updated template (NOTE: forward slashes only, should start with a slash)
	 */
	public function updateTemplate(
		string $environment,
		string $templateId,
		?string $name,
		?string $folderPath,
	): Response
	{
		return $this->connector->send(new UpdateTemplate($environment, $templateId, $name, $folderPath));
	}


	/**
	 * @param string $environment Name of the environment
	 * @param string $templateId ID of the template
	 * @param bool $force When set to 'true' template is removed from all collections
	 */
	public function deleteTemplate(string $environment, string $templateId, ?bool $force): Response
	{
		return $this->connector->send(new DeleteTemplate($environment, $templateId, $force));
	}


	/**
	 * @param string $environment Name of the environment
	 * @param string $templateId ID of the template
	 */
	public function downloadTemplate(string $environment, string $templateId): Response
	{
		return $this->connector->send(new DownloadTemplate($environment, $templateId));
	}


	/**
	 * @param string $environment Name of the environment
	 * @param string $templateId ID of the template
	 * @param string $previewType Type of preview
	 */
	public function getTemplatePreview(
		string $environment,
		string $templateId,
		?string $previewType,
	): Response
	{
		return $this->connector->send(new GetTemplatePreview($environment, $templateId, $previewType));
	}
}
