<?php

namespace Ptb\CartRover\Tests\Feature;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Ptb\CartRover\Actions\CancelOrder;
use Ptb\CartRover\Actions\NewOrder;
use Ptb\CartRover\Actions\NewOrders;
use Ptb\CartRover\Actions\UpdateOrderStatus;
use Ptb\CartRover\Actions\ViewInventory;
use Ptb\CartRover\Actions\ViewInventoryByProduct;
use Ptb\CartRover\Actions\ViewOrder;
use Ptb\CartRover\Actions\ViewOrdersByStatus;
use Ptb\CartRover\Actions\ViewOrderStatus;
use Ptb\CartRover\Models\Order;
use Ptb\CartRover\Tests\TestCase;

class PackageTest extends TestCase
{
    /** @test */
    public function can_load_config()
    {
        $config = config('cart-rover-cart');

        $this->assertArrayHasKey('api_username', $config);
        $this->assertArrayHasKey('api_key', $config);
    }

    /** @test */
    public function can_cancel_order()
    {
        $response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/CancelOrder_success.json'), true);
        Http::fake([
            config('cart-rover-cart.api_url') . '/*' => Http::response($response, 200),
        ]);

        $reference = 'TEST01';
        $result = (new CancelOrder)->handle($reference);

        $this->assertArrayHasKey('success_code', $result);
    }

    /** @test */
    public function can_create_order()
    {
        $response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/NewOrder_success.json'), true);
        Http::fake([
            config('cart-rover-cart.api_url') . '/*' => Http::response($response, 200),
        ]);

        $request = json_decode(file_get_contents(__DIR__ . '/../Fixtures/Requests/NewOrder.json'), true);
        $result = (new NewOrder)->handle(Order::make($request));

        $this->assertArrayHasKey('success_code', $result);
    }

    /** @test */
    public function can_create_orders()
    {
        $response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/NewOrder_success.json'), true);
        Http::fake([
            config('cart-rover-cart.api_url') . '/*' => Http::response($response, 200),
        ]);

        $request = json_decode(file_get_contents(__DIR__ . '/../Fixtures/Requests/NewOrders.json'), true);
        $result = (new NewOrders)->handle(collect($request));

        $this->assertArrayHasKey('success_code', $result);
    }

    /** @test */
    public function can_update_order_status()
    {
        $response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/UpdateOrderStatus_success.json'), true);
        Http::fake([
            config('cart-rover-cart.api_url') . '/*' => Http::response($response, 200),
        ]);

        $reference = 'TEST01';
        $result = (new UpdateOrderStatus)->handle($reference, Order::StatusConfirmed, []);

        $this->assertArrayHasKey('success_code', $result);
    }

    /** @test */
    public function can_view_order()
    {
        $view_order_response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/ViewOrder_success.json'), true);
        Http::fake([
            config('cart-rover-cart.api_url') . '/*' => Http::response($view_order_response, 200),
        ]);

        $reference = 'TEST07Auto';
        $order = (new ViewOrder)->handle($reference);

        $this->assertInstanceOf(Order::class, $order);
    }

    /** @test */
    public function can_view_orders_by_status()
    {
        $view_order_response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/ViewOrdersByStatus_success.json'), true);
        Http::fake([
            config('cart-rover-cart.api_url') . '/*' => Http::response($view_order_response, 200),
        ]);

        $orders = (new ViewOrdersByStatus)->handle(Order::StatusAtWms);

        $this->assertInstanceOf(Collection::class, $orders);
    }

    /** @test */
    public function can_view_order_status()
    {
        $view_order_response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/ViewOrderStatus_success.json'), true);
        Http::fake([
            config('cart-rover-cart.api_url') . '/*' => Http::response($view_order_response, 200),
        ]);

        $reference = '349347';
        $result = (new ViewOrderStatus)->handle($reference);

        $this->assertArrayHasKey('success_code', $result);
        $this->assertArrayHasKey('response', $result);
        $this->assertTrue($result['success_code']);
        $this->assertEquals($reference, $result['response']['cust_ref']);
    }

    /** @test */
    public function can_view_inventory()
    {
        $response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/ViewInventory_success.json'), true);
        Http::fake([
            str_replace('v1/cart', 'v1/merchant', config('cart-rover-cart.api_url')) . '/*' => Http::response($response, 200),
        ]);

        $result = (new ViewInventory())->handle();

        $this->assertArrayHasKey('success_code', $result);
        $this->assertArrayHasKey('response', $result);
        $this->assertTrue($result['success_code']);
        $this->assertCount(3, $result['response']);
    }

    /** @test */
    public function can_view_inventory_by_product()
    {
        $response = json_decode(file_get_contents(__DIR__ . '/../Fixtures/ViewInventoryByProduct_success.json'), true);
        Http::fake([
            str_replace('v1/cart', 'v1/merchant', config('cart-rover-cart.api_url')) . '/*' => Http::response($response, 200),
        ]);

        $sku = 'CIO-1001';
        $result = (new ViewInventoryByProduct())->handle($sku);
        $this->assertArrayHasKey('success_code', $result);
        $this->assertArrayHasKey('response', $result);
        $this->assertTrue($result['success_code']);
        $this->assertEquals($sku, $result['response']['sku']);
    }
}