<?php

namespace Ptb\Accuzip\Actions;

use Illuminate\Support\Facades\Storage;
use Ptb\Accuzip\AccuzipConnector;
use Ptb\Accuzip\Events\TaskCompleted;
use Ptb\Accuzip\Jobs\CASS\Certify as CASS;
use Ptb\Accuzip\Jobs\CASS_DUPS_PRESORT\Process as CASS_DUPS_PRESORT;
use Ptb\Accuzip\Jobs\CASS_NCOA_DUPS_PRESORT\Process as CASS_NCOA_DUPS_PRESORT;
use Ptb\Accuzip\Jobs\CASS_NCOA_PRESORT\Process as CASS_NCOA_PRESORT;
use Ptb\Accuzip\Jobs\CASS_PRESORT\Process as CASS_PRESORT;
use Ptb\Accuzip\Jobs\DUPS\Detect as DUPS;
use Ptb\Accuzip\Jobs\NCOA\Certify as NCOA;
use Ptb\Accuzip\Jobs\PRESORT\Process as PRESORT;
use Ptb\Accuzip\Models\AccuzipQuote;
use Ptb\Accuzip\Requests\QUOTE\GetQuote;
use Ptb\Accuzip\Requests\QUOTE\PutQuote;
use Ptb\Accuzip\Requests\QUOTE\PostUpload;
use ReflectionException;
use Saloon\Contracts\Response;
use Saloon\Exceptions\InvalidResponseClassException;
use Saloon\Exceptions\PendingRequestException;

final class UploadAndProcessMailFile
{
    /**
     * @param AccuzipConnector $service
     */
    public function __construct(
        private readonly AccuzipConnector $service,
    )
    {}

    /**
     * @param AccuzipQuote $quote
     * @throws InvalidResponseClassException
     * @throws PendingRequestException
     * @throws ReflectionException
     */
    public function handle(AccuzipQuote $quote): void
    {
        logger()->debug('Starting accuzip upload');
        $response = $this->service->send(request: $this->buildUploadRequest($quote));
        if ($response->failed()) {
            $response->throw();
        }

        logger()->debug('Upload complete. Updating local quote');
        $quote->update([
            'guid' => $response->json('guid'),
            'upload_complete' => true
        ]);

        logger()->debug('Update complete. Triggering TaskCompleted event');
        event(new TaskCompleted(
            guid: $response->json('guid'),
            taskName: 'UPLOAD'
        ));

        logger()->debug('Sending get quote request');
        $response = $this->sendQuoteRequest(
            request: $this->buildGetQuoteRequest($quote),
        );

        logger()->debug('Get quote complete, updating local quote');
        $quote->update([
            'total_records' => $response->json('total_records') ?? 0,
        ]);

        logger()->debug('Sending update quote request');
        $response = $this->service->send(request: $this->buildUpdateQuoteRequest($quote));
        if ($response->failed()) {
            $response->throw();
        }

        $quote->fresh();
        logger()->debug('Update quote request complete, triggering job');
        switch (true) {
            case $quote->template->cassEnabled && $quote->template->ncoaEnabled && $quote->template->duplicatesEnabled && $quote->template->presortWnabled:
                logger()->debug('CASS-NCOA-DUPS-PRESORT');
                // CASS-NCOA-DUPS-PRESORT
                CASS_NCOA_DUPS_PRESORT::dispatch(quote: $quote);
                break;
            case $quote->template->cassEnabled && $quote->template->ncoaEnabled && $quote->template->presortEnabled:
                logger()->debug('CASS-NCOA-PRESORT');
                // CASS-NCOA-PRESORT
                CASS_NCOA_PRESORT::dispatch(quote: $quote);
                break;
            case $quote->template->cassEnabled && $quote->template->duplicatesEnabled && $quote->template->presortEnabled:
                logger()->debug('CASS-DUPS-PRESORT');
                // CASS-DUPS-PRESORT
                CASS_DUPS_PRESORT::dispatch(quote: $quote);
                break;
            case $quote->template->cassEnabled && $quote->template->presortEnabled:
                logger()->debug('CASS-PRESORT');
                // CASS-PRESORT
                CASS_PRESORT::dispatch(quote: $quote);
                break;
            case $quote->template->cassEnabled:
                logger()->debug('CASS');
                // CASS
                CASS::dispatch(quote: $quote);
                break;
            case $quote->template->ncoaEnabled:
                logger()->debug('NCOA');
                // NCOA
                NCOA::dispatch(quote: $quote);
                break;
            case $quote->template->duplicatesEnabled:
                logger()->debug('DUPS');
                // DUPS
                DUPS::dispatch(quote: $quote);
                break;
            case $quote->template->presortEnabled:
                logger()->debug('PRESORT');
                // PRESORT
                PRESORT::dispatch(quote: $quote);
                break;
            default:
                logger()->debug('NO JOB MATCH FOUND');
                logger()->debug('quote', $quote->toArray());
                break;
        }
    }

    /**
     * @param GetQuote $request
     * @return Response
     * @throws InvalidResponseClassException
     * @throws PendingRequestException
     * @throws ReflectionException
     */
    protected function sendQuoteRequest(GetQuote $request): Response
    {
        $response = $this->service->send(request: $request);
        if ($response->failed()) {
            $response->throw();
        }

        while ($response->json('success') === false) {
            sleep(2);
            $response = $this->service->send(request: $request);
            if ($response->failed()) {
                $response->throw();
            }
        }

        return $response;
    }

    /**
     * @param AccuzipQuote $quote
     * @return PostUpload
     */
    protected function buildUploadRequest(AccuzipQuote $quote): PostUpload
    {
        return new PostUpload(
            filename: basename($quote->source_filepath),
            data: Storage::disk($quote->source_disk)->get($quote->source_filepath),
            callbackUrl: route('accuzip.callback'),
            listOwnerPafId: config('accuzip.paf_id'),
        );
    }

    /**
     * @param AccuzipQuote $quote
     * @return GetQuote
     */
    public function buildGetQuoteRequest(AccuzipQuote $quote): GetQuote
    {
        return new GetQuote(guid: $quote->guid);
    }

    /**
     * @param AccuzipQuote $quote
     * @return PutQuote
     */
    protected function buildUpdateQuoteRequest(AccuzipQuote $quote): PutQuote
    {
        return new PutQuote(quote: $quote);
    }
}