<?php

namespace Ptb\Accuzip\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Ptb\Accuzip\AccuzipConnector;
use Ptb\Accuzip\Events\TaskCompleted;
use Ptb\Accuzip\Models\AccuzipQuote;
use Ptb\Accuzip\Requests\QUOTE\GetQuote;
use ReflectionException;
use Saloon\Contracts\Response;
use Saloon\Exceptions\InvalidResponseClassException;
use Saloon\Exceptions\PendingRequestException;

class AccuzipController extends Controller
{
    /**
     * @param Request $request
     * @param AccuzipConnector $service
     * @return JsonResponse
     * @throws ReflectionException
     * @throws InvalidResponseClassException
     * @throws PendingRequestException
     */
    public function __invoke(
        Request $request,
        AccuzipConnector $service,
    )
    {
        $guid = $request->get('guid');
        if (empty($guid)) {
            abort(404);
        }

        $quote = AccuzipQuote::query()->whereGuid($guid)->firstOrFail();
        $response = $service->send(
            request: $this->buildQuoteRequest(quote: $quote)
        );

        if ($response->failed()) {
            $response->throw();
        }

        // update our internal record
        $task_process = $response->json('task_process');
        switch ($task_process) {
            case 'CASS':
                $this->handleCassCompletion($quote, $response);
                break;
            case 'NCOA':
                $this->handleNcoaCompletion($quote, $response);
                break;
            case 'DUPS':
                $this->handleDuplicateCompletion($quote, $response);
                break;
            case 'PRESORT':
                $this->handlePresortCompletion($quote, $response);
                break;
            case 'CASS-NCOA-PRESORT': // All-In-One
                $this->handleCassCompletion($quote, $response);
                $this->handleNcoaCompletion($quote, $response);
                $this->handlePresortCompletion($quote, $response);
                break;
            case 'CASS-NCOA-DUPS-PRESORT': // All-In-One
                $this->handleCassCompletion($quote, $response);
                $this->handleNcoaCompletion($quote, $response);
                $this->handleDuplicateCompletion($quote, $response);
                $this->handlePresortCompletion($quote, $response);
                break;
            default:
                logger()->error('No matching task process');
                logger()->debug('quote', $response->json());
                logger()->debug($task_process);
                break;
        }

        return response()->json();
    }

    /**
     * @param AccuzipQuote $quote
     * @param Response $response
     * @return void
     */
    protected function handleCassCompletion(AccuzipQuote $quote, Response $response)
    {
        $quote->update([
            'cass_certification_complete' => true
        ]);

        event(new TaskCompleted(
            guid: $quote->guid,
            taskName: 'CASS'
        ));
    }

    /**
     * @param AccuzipQuote $quote
     * @param Response $response
     * @return void
     */
    protected function handleNcoaCompletion(AccuzipQuote $quote, Response $response)
    {
        $quote->update([
            'ncoa_link_certification_complete' => true
        ]);

        event(new TaskCompleted(
            guid: $quote->guid,
            taskName: 'NCOA'
        ));
    }

    /**
     * @param AccuzipQuote $quote
     * @param Response $response
     * @return void
     */
    protected function handleDuplicateCompletion(AccuzipQuote $quote, Response $response)
    {
        $quote->update([
            'duplicate_detection_complete' => true
        ]);

        event(new TaskCompleted(
            guid: $quote->guid,
            taskName: 'DUPS'
        ));
    }

    /**
     * @param AccuzipQuote $quote
     * @param Response $response
     * @return void
     */
    public function handlePresortCompletion(AccuzipQuote $quote, Response $response)
    {
        $postage = preg_replace('/[^0-9.]/', '', $response->json('total_postage'));
        if (empty($postage)) {
            $postage = 0;
        }

        $postage_saved = preg_replace('/[^0-9.]/', '', $response->json('postage_saved'));
        if (empty($postage_saved)) {
            $postage_saved = 0;
        }

        $quote->update([
            'presort_complete' => true,
            'total_records' => (int) $response->json('total_records'),
            'total_presort_records' => (int) $response->json('total_presort_records'),
            'postage' => $postage,
            'postage_saved' => $postage_saved,
        ]);

        event(new TaskCompleted(
            guid: $quote->guid,
            taskName: 'PRESORT'
        ));

        logger()->debug('quote finished', $response->json());
    }

    /**
     * @param AccuzipQuote $quote
     * @return GetQuote
     */
    protected function buildQuoteRequest(AccuzipQuote $quote): GetQuote
    {
        return new GetQuote(guid: $quote->guid);
    }
}
