<?php

namespace Ptb\Accuzip\Models;

use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Ptb\Accuzip\Actions\UploadAndProcessMailFile;
use Ptb\Accuzip\Data\ColumnMapData;
use Ptb\Accuzip\Data\MailingAgentData;
use Ptb\Accuzip\Data\PermitHolderData;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Ptb\Accuzip\AccuzipConnector;
use Ptb\Accuzip\Data\PreparedForData;
use Ptb\Accuzip\Data\QuoteTemplateData;
use Ptb\Accuzip\Enums\QUOTE\MailPieceSize;
use Ptb\Accuzip\Enums\QUOTE\PresortClass;
use Ptb\Accuzip\Enums\QUOTE\TrayType;
use ReflectionException;
use Saloon\Exceptions\InvalidResponseClassException;
use Saloon\Exceptions\PendingRequestException;

class AccuzipQuote extends Model
{
    use HasFactory;

    protected $guarded = [];

    protected $casts = [
        'total_records' => 'integer',
        'total_presort_records' => 'integer',
        'upload_complete' => 'boolean',
        'do_cass' => 'boolean',
        'cass_certification_complete' => 'boolean',
        'do_ncoa' => 'boolean',
        'ncoa_link_certification_complete' => 'boolean',
        'do_dups' => 'boolean',
        'duplicate_detection_complete' => 'boolean',
        'do_presort' => 'boolean',
        'presort_complete' => 'boolean',
        'mail_date' => 'date',
        'postage' => 'decimal:2',
        'postage_saved' => 'decimal:2',
        'template' => QuoteTemplateData::class,
    ];

    /**
     * Upload mail file to Accuzip API to begin processing
     *
     * @return void
     * @throws ReflectionException
     * @throws InvalidResponseClassException
     * @throws PendingRequestException
     */
    public function upload(): void
    {
        app(UploadAndProcessMailFile::class)->handle(
            quote: $this
        );
    }

    /**
     * Download preview file. Useful for further integration with CHILI Publish, etc
     *
     * @return string
     */
    public function downloadCSVPreview(): string
    {
        $service = app(AccuzipConnector::class);
        $url = "{$service->resolveBaseUrl()}/ws_360_webapps/download.jsp?guid={$this->guid}&ftype=prev.csv";
        $path = "{$this->guid}/maillist-preview.csv";
        $success = Storage::disk($this->source_disk)->put(
            path: $path,
            contents: file_get_contents($url),
            options: [
                'public'
            ]
        );

        if ($success) {
            $this->update(['print_ready_filepath' => $path]);

            return Storage::disk($this->source_disk)->url($path);
        }

        return false;
    }

    /**
     * Download clean address CSV
     *
     * @return string
     */
    public function downloadCSV(): string
    {
        $service = app(AccuzipConnector::class);
        $url = "{$service->resolveBaseUrl()}/ws_360_webapps/download.jsp?guid={$this->guid}&ftype=csv";
        $path = "{$this->guid}/maillist.csv";
        $success = Storage::disk($this->source_disk)->put(
            path: $path,
            contents: file_get_contents($url),
            options: [
                'public'
            ]
        );

        if ($success) {
            $this->update(['print_ready_filepath' => $path]);

            return Storage::disk($this->source_disk)->url($path);
        }

        return false;
    }

    /**
     * Download USPS eDocumentation PDF
     *
     * @return string
     */
    public function downloadPDF(): string
    {
        $service = app(AccuzipConnector::class);
        $url = "{$service->resolveBaseUrl()}/ws_360_webapps/download.jsp?guid={$this->guid}&ftype=pdf";
        $path = "{$this->guid}/mail.PDF";
        $success = Storage::disk($this->source_disk)->put(
            path: $path,
            contents: file_get_contents($url),
            options: [
                'public'
            ]
        );

        if ($success) {
            $this->update(['usps_pdf_filepath' => $path]);


            return Storage::disk($this->source_disk)->url($path);
        }

        return false;
    }

    /**
     * Download ZIP file containing all mail.dat contents
     *
     * @return string
     */
    public function downloadMailDat(): string
    {
        $service = app(AccuzipConnector::class);
        $url = "{$service->resolveBaseUrl()}/ws_360_webapps/download.jsp?guid={$this->guid}&ftype=maildat.zip";
        $path = "{$this->guid}/maildat.ZIP";
        $success = Storage::disk($this->source_disk)->put(
            path: $path,
            contents: file_get_contents($url),
            options: [
                'public'
            ]
        );

        if ($success) {
            $this->update(['maildat_filepath' => $path]);
            return Storage::disk($this->source_disk)->url($path);
        }

        return false;
    }
}
